<?php

namespace App\Http\Controllers;

use App\Exports\TasksExport;
use App\Models\Assignee;
use App\Models\Attachment;
use App\Models\Background;
use App\Models\BoardList;
use App\Models\CheckList;
use App\Models\Comment;
use App\Models\Label;
use App\Models\Project;
use App\Models\RecentProject;
use App\Models\Setting;
use App\Models\StarredProject;
use App\Models\Task;
use App\Models\TaskLabel;
use App\Models\TeamMember;
use App\Models\Timer;
use App\Models\Workspace;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Response;
use Inertia\Inertia;
use Maatwebsite\Excel\Excel;

class ProjectsController extends Controller {
    public function index(){
        return Inertia::render('Projects/Index', [
            'title' => 'Projects',
        ]);
    }

    public function test(){
        return Inertia::render('Projects/Test', [
            'title' => 'Projects',
        ]);
    }

    public function uploadBackground($id, Request $request)
    {
        $background = null;
        if($request->file('file')){
            $file = $request->file('file');
            $allowedMimeTypes = [ 'image/jpeg','image/gif','image/png', 'image/svg+xml', 'image/webp' ];
            $contentType = $file->getClientmimeType();

            if(! in_array($contentType, $allowedMimeTypes) ){
                return response()->json(['error' => true, 'message' => 'File type is not supported!']);
            }
            $file_name_origin = $file->getClientOriginalName();
            $file_name = uniqid().'-'.$this->clean(pathinfo($file_name_origin, PATHINFO_FILENAME)).'.'.$file->getClientOriginalExtension();
            $file_path = '/files/'.$file->storeAs('projects', $file_name, ['disk' => 'file_uploads']);
            $background = Background::create(['image' => $file_path, 'bg' => '#624b66', 'top' => '#624b66', 'side' => '#705675e6', 'type' => 'custom']);
        }
        return response()->json($background);
    }

    public function updateBackground($projectId, Request $request)
    {
        $background = null;
        $requests = $request->all();
        if(!empty($requests['background_id'])){
            $background = Background::where('id', $requests['background_id'])->first();
            if(!empty($background)){
                $project = Project::where('id', $projectId)->first();
                if(!empty($project)){
                    if(!empty($project->background_id)){
                        $previous_background = Background::where('id', $project->background_id)->first();
                        if(!empty($previous_background) && $previous_background->type == 'custom'){
                            if(!empty($previous_background->image) && File::exists(public_path($previous_background->image))){
                                File::delete(public_path($previous_background->image));
                            }
                            $previous_background->delete();
                        }
                    }
                    $project->background_id = $background->id;
                    $project->save();
                }
            }
        }
        return response()->json($background);
    }

    public function jsonCreate(Request $request){
        $requests = $request->all();
        $requests['user_id'] = auth()->id();

        $project = Project::create($requests);

        $enable_list = Setting::where('slug', 'enable_pre_made_board')->first();
        if(!empty($enable_list) && $enable_list->value){
            $board_list = Setting::where('slug', 'pre_made_board_list')->first();
            if(!empty($board_list)){
                $list_items = is_string($board_list->value) ? json_decode($board_list->value, true) : $board_list->value;
                $io = 0;
                foreach ($list_items as $item){
                    BoardList::create(['user_id' => $requests['user_id'], 'order' => $io, 'project_id' => $project->id, 'title' => $item]);
                    $io += 1;
                }
            }
        }

        $slug = $this->clean($project->title);
        $existingItem = Project::where('slug', $slug)->first();
        if(!empty($existingItem)){
            $slug = $slug . '-' . $project->id;
        }
        $project->slug = $slug;
        $project->save();

        return response()->json($project);
    }

    public function jsonMembers($project_id){
        $assignees = Assignee::whereHas('task', function ($q) use ($project_id) {
            $q->where('project_id', $project_id);
        })->where('user_id', '!=', auth()->id())->groupBy('user_id')->with('user:id,first_name,last_name,photo_path')->get();
        return response()->json($assignees);
    }

    public function jsonFilterData($project_id){
        $assignees = Assignee::whereHas('task', function ($q) use ($project_id) {
            $q->where('project_id', $project_id);
        })->where('user_id', '!=', auth()->id())->groupBy('user_id')->with('user:id,first_name,last_name,photo_path')->get();
        $labels = Label::where('project_id', $project_id)->orderBy('name')->get();
        return response()->json(['assignees' => $assignees, 'labels' => $labels]);
    }

    public function all(){
        $projects = Project::get();
        return response()->json($projects);
    }

    public function jsonAll($workspace_id){
        $projects = Project::where('workspace_id', $workspace_id)->with('background')->with('star')->get();
        return response()->json($projects);
    }

    public function jsonRecent(){
        $user_id = auth()->id();
        $workspaceIds = Workspace::where('user_id', $user_id)->orWhereHas('member')->pluck('id');
        $projects = RecentProject::where('user_id', $user_id)->with('project')->has('project.workspace')->whereHas('project', function ($q) use ($workspaceIds) {
            $q->whereIn('workspace_id', $workspaceIds);
        })->orderBy('opened', 'desc')->paginate(10)
            ->through(function ($project) {
                return [
                    'id' => $project->project->id,
                    'title' => $project->project->title,
                    'slug' => $project->project->slug,
                    'star' => (bool)$project->project->star,
                    'workspace' => $project->project->workspace->name,
                    'background' => $project->project->background?$project->project->background->image:null,
                ];
            });
        return response()->json($projects);
    }

    public function jsonStar(){
        $user_id = auth()->id();
        $workspaceIds = Workspace::where('user_id', $user_id)->orWhereHas('member')->pluck('id');
        $projects = StarredProject::where('user_id', $user_id)->with('project')->has('project.workspace')->whereHas('project', function ($q) use ($workspaceIds) {
            $q->whereIn('workspace_id', $workspaceIds);
        })->orderBy('updated_at', 'desc')->paginate(100)
            ->through(function ($project) {
                return [
                    'id' => $project->project->id,
                    'title' => $project->project->title,
                    'slug' => $project->project->slug,
                    'star' => (bool)$project->project->star,
                    'workspace' => $project->project->workspace->name,
                    'background' => $project->project->background?$project->project->background->image:null,
                ];
            });
        return response()->json($projects);
    }

    public function update($id, Request $request){
        $project = Project::whereId($id)->first();
        $requestData = $request->all();
        foreach ($requestData as $itemKey => $itemValue){
            $project->{$itemKey} = $itemValue;
        }

        $slug = $this->clean($project->title);
        $existingItem = Project::where('id', '!=', $project->id)->where('slug', $slug)->first();
        if(!empty($existingItem)){
            $slug = $slug . '-' . $project->id;
        }
        $project->slug = $slug;

        $project->save();
        return response()->json($project);
    }

    public function noProject(){
        return Inertia::render('Projects/Na', [
            'title' => 'No Workspace',
            'notice' => 'You did not assigned any workspace yet. Please contact with admin'
        ]);
    }

    public function view($uid, Request $request){
        $auth_id = auth()->id();
        $workspaceIds = Workspace::where('user_id', $auth_id)->orWhereHas('member')->pluck('id');
        $requests = $request->all();
        $project = Project::bySlugOrId($uid)->whereIn('workspace_id', $workspaceIds)->with('workspace.member')->with('star')->with('background')->first();
        if(empty($project)){
            return abort(404);
        }
        RecentProject::updateOrCreate(['user_id' => $auth_id, 'project_id' => $project->id], ['opened' => Carbon::now()]);
        $list_index = [];
        $board_lists = BoardList::where('project_id', $project->id)->isOpen()->orderByOrder()->get()->toArray();
        $loopIndex = 0;
        foreach ($board_lists as &$listItem){
            $list_index[$listItem['id']] = $loopIndex;
            $listItem['tasks'] = [];
            $loopIndex+= 1;
        }
        if($project->is_private && (auth()->user()['role_id'] != 1)){
            $requests['private_task'] = $auth_id;
        }
        $tasks = Task::filter($requests)
            ->isOpen()
            ->byProject($project->id)
            ->with('taskLabels.label')
            ->with('timer')
            ->whereHas('list')
            ->with('cover')
            ->withCount('checklistDone')
            ->withCount('comments')
            ->withCount('checklists')
            ->withCount('attachments')->with('assignees')
            ->orderByOrder()->get()->toArray();
        foreach ($tasks as $task){
            if(isset($list_index[$task['list_id']])){
                $board_lists[$list_index[$task['list_id']]]['tasks'][] = $task;
            }
        }
        return Inertia::render('Projects/View', [
            'title' => 'Board | '.$project->title,
            'board_lists' => $board_lists,
            'lists' => $board_lists,
            'list_index' => $list_index,
            'filters' => $requests,
            'project' => $project,
            'tasks' => $tasks,
        ]);
    }

    public function viewWithTask($projectUid, $taskUid, Request $request){
        $requests = $request->all();
        $auth_id = auth()->id();
        $workspaceIds = Workspace::where('user_id', $auth_id)->orWhereHas('member')->pluck('id');
        $project = Project::bySlugOrId($projectUid)->whereIn('workspace_id', $workspaceIds)->with('workspace.member')->with('star')->with('background')->first();
        $list_index = [];
        $board_lists = BoardList::where('project_id', $project->id)->isOpen()->orderByOrder()->get()->toArray();
        $loopIndex = 0;
        foreach ($board_lists as &$listItem){
            $list_index[$listItem['id']] = $loopIndex;
            $listItem['tasks'] = [];
            $loopIndex+= 1;
        }
        $tasks = Task::filter($requests)
            ->isOpen()
            ->byProject($project->id)
            ->with('taskLabels.label')
            ->whereHas('list')
            ->withCount('checklistDone')
            ->withCount('comments')
            ->withCount('checklists')
            ->withCount('attachments')
            ->with('assignees')
            ->orderByOrder()
            ->get()->toArray();
        foreach ($tasks as $task){
            if(isset($list_index[$task['list_id']])){
                $board_lists[$list_index[$task['list_id']]]['tasks'][] = $task;
            }
        }
        return Inertia::render('Projects/View', [
            'title' => 'Projects',
            'filters' => $requests,
            'board_lists' => $board_lists,
            'lists' => $board_lists,
            'list_index' => $list_index,
            'project' => $project,
            'task' => Task::where('id', $taskUid)->orWhere('slug', $taskUid)->first(),
            'tasks' => $tasks,
        ]);
    }

    public function viewTable($uid, Request $request){
        $requests = $request->all();
        $auth_id = auth()->id();
        $workspaceIds = Workspace::where('user_id', $auth_id)->orWhereHas('member')->pluck('id');
        $project = Project::bySlugOrId($uid)->whereIn('workspace_id', $workspaceIds)->with('workspace.member')->with('star')->with('background')->first();
        $list_index = [];
        $board_lists = BoardList::where('project_id', $project->id)->isOpen()->orderByOrder()->get()->toArray();
        $loopIndex = 0;
        foreach ($board_lists as &$listItem){
            $list_index[$listItem['id']] = $loopIndex;
            $listItem['tasks'] = [];
            $loopIndex+= 1;
        }
        $tasks = Task::filter($requests)
            ->isOpen()
            ->byProject($project->id)
            ->with('taskLabels.label')
            ->with('timer')
            ->whereHas('list')
            ->with('assignees')
            ->with('list')
            ->orderByOrder()
            ->get()->toArray();
        foreach ($tasks as $task){
            if(isset($list_index[$task['list_id']])){
                $board_lists[$list_index[$task['list_id']]]['tasks'][] = $task;
            }
        }
        return Inertia::render('Projects/Table', [
            'title' => 'Table | '.$project->title,
            'board_lists' => $board_lists,
            'lists' => $board_lists,
            'list_index' => $list_index,
            'project' => $project,
            'filters' => $requests,
            'tasks' => $tasks
        ]);
    }

    public function viewTableWithTask($uid, $taskUid, Request $request){
        $requests = $request->all();
        $auth_id = auth()->id();
        $workspaceIds = Workspace::where('user_id', $auth_id)->orWhereHas('member')->pluck('id');
        $project = Project::bySlugOrId($uid)->whereIn('workspace_id', $workspaceIds)->with('workspace.member')->with('star')->with('background')->first();
        $list_index = [];
        $board_lists = BoardList::where('project_id', $project->id)->isOpen()->orderByOrder()->get()->toArray();
        $loopIndex = 0;
        foreach ($board_lists as &$listItem){
            $list_index[$listItem['id']] = $loopIndex;
            $listItem['tasks'] = [];
            $loopIndex+= 1;
        }
        $tasks = Task::filter($requests)
            ->isOpen()
            ->byProject($project->id)
            ->with('taskLabels.label')
            ->whereHas('list')
            ->with('assignees')
            ->with('list')
            ->orderByOrder()
            ->get()->toArray();
        foreach ($tasks as $task){
            if(isset($list_index[$task['list_id']])){
                $board_lists[$list_index[$task['list_id']]]['tasks'][] = $task;
            }
        }
        return Inertia::render('Projects/Table', [
            'title' => 'Projects',
            'board_lists' => $board_lists,
            'lists' => $board_lists,
            'list_index' => $list_index,
            'filters' => $requests,
            'project' => $project,
            'task' => Task::where('id', $taskUid)->orWhere('slug', $taskUid)->first(),
            'timer' => Timer::with('task')->mine()->running()->first() ?? null,
            'tasks' => $tasks
        ]);
    }

    public function viewDashboard($uid){
        $auth_id = auth()->id();
        $workspaceIds = Workspace::where('user_id', $auth_id)->orWhereHas('member')->pluck('id');
        $project = Project::bySlugOrId($uid)->whereIn('workspace_id', $workspaceIds)->with('workspace.member')->with('star')->with('background')->first();
        $taskIds = Task::where('project_id', $project->id)->pluck('id')->toArray();
        $per_list = Task::select('list_id', DB::raw('count(*) as total'))->where('project_id', $project->id)->groupBy('list_id')->whereHas('list')->with('list')->get()->toArray();
        $per_assignee = Assignee::select('user_id', DB::raw('count(*) as total'))->whereIn('task_id', $taskIds)->groupBy('user_id')->with('user')->get()->toArray();
        $per_label = TaskLabel::select('label_id', DB::raw('count(*) as total'))->whereIn('task_id', $taskIds)->groupBy('label_id')->with('label')->get()->toArray();
        $due_done = Task::where('project_id', $project->id)->where('is_done', 1)->count();
        $no_due = Task::where('project_id', $project->id)->whereNull('due_date')->count();
        $due_over = Task::where('project_id', $project->id)->where('due_date', '<', Carbon::now())->count();
        $due_later = Task::where('project_id', $project->id)->where('due_date', '>', Carbon::now()->addDay())->count();
        $due_soon = Task::where('project_id', $project->id)->whereBetween('due_date', [Carbon::now(), Carbon::now()->addDay()])->count();
        return Inertia::render('Projects/Dashboard', [
            'title' => 'Dashboard | '.$project->title,
            'per_list' => $per_list,
            'project' => $project,
            'per_assignee' => $per_assignee,
            'per_label' => $per_label,
            'due_data' => [
                ['due' => ['name' => 'Complete', 'color' => '#22A06B' ], 'total' => $due_done],
                ['due' => ['name' => 'Due soon', 'color' => '#B38600' ], 'total' => $due_soon],
                ['due' => ['name' => 'Due later', 'color' => '#E56910' ], 'total' => $due_later],
                ['due' => ['name' => 'Overdue', 'color' => '#C9372C' ], 'total' => $due_over],
                ['due' => ['name' => 'No due date', 'color' => '#607d8b' ], 'total' => $no_due],
            ]
        ]);
    }

    public function viewCalendar($uid, Request $request)
    {
        $requests = $request->all();
        $auth_id = auth()->id();
        $workspaceIds = Workspace::where('user_id', $auth_id)->orWhereHas('member')->pluck('id');
        $project = Project::bySlugOrId($uid)->whereIn('workspace_id', $workspaceIds)->with('workspace.member')->with('star')->with('background')->first();
        $list_index = [];
        $board_lists = BoardList::where('project_id', $project->id)->isOpen()->orderByOrder()->get()->toArray();
        $loopIndex = 0;
        foreach ($board_lists as &$listItem){
            $list_index[$listItem['id']] = $loopIndex;
            $listItem['tasks'] = [];
            $loopIndex+= 1;
        }
        $tasks = Task::filter($requests)
            ->isOpen()
            ->byProject($project->id)
            ->with('taskLabels.label')
            ->with('timer')
            ->whereHas('list')
            ->with('assignees')
            ->with('list')
            ->orderByOrder()
            ->get()->toArray();
        foreach ($tasks as $task){
            if(isset($list_index[$task['list_id']])){
                $board_lists[$list_index[$task['list_id']]]['tasks'][] = $task;
            }
        }
        return Inertia::render('Projects/Calendar', [
            'title' => 'Calendar | '.$project->title,
            'board_lists' => $board_lists,
            'lists' => $board_lists,
            'list_index' => $list_index,
            'project' => $project,
            'filters' => $requests,
            'tasks' => $tasks
        ]);
    }

    public function viewTimeLogs($projectUid, Request $request){
        $requests = $request->all();
        $auth_id = auth()->id();
        $workspaceIds = Workspace::where('user_id', $auth_id)->orWhereHas('member')->pluck('id');
        $project = Project::bySlugOrId($projectUid)->whereIn('workspace_id', $workspaceIds)->with('workspace.member')->with('star')->with('background')->first();
        $timerQuery = Timer::whereHas('task', function ($q) use ($project) {
            $q->where('project_id', $project->id);
        })->filter($requests);
        return Inertia::render('Projects/Timer', [
            'title' => 'Time Logs | '.$project->title,
            'project' => $project,
            'filters' => $requests,
            'total_duration' => $timerQuery->sum('duration'),
            'time_logs' => $timerQuery->with('task')
                ->with('user')
                ->orderBy('created_at', 'DESC')
                ->paginate(9)
                ->withQueryString()
                ->through(function ($log) {
                    return [
                        'id' => $log->id,
                        'title' => $log->title,
                        'user' => $log->user,
                        'task' => $log->task,
                        'task_id' => $log->task_id,
                        'duration' => $log->duration,
                        'started_at' => $log->started_at,
                        'stopped_at' => $log->stopped_at,
                        'created_at' => $log->created_at,
                    ];
                } ),
        ]);
    }

    public function projectOtherData($project_id){
        $project = Project::where('id', $project_id)->first();
        $labels = Label::where('project_id', $project_id)->get();
        $lists = BoardList::withCount('tasks')->get();
        $teamMembers = TeamMember::with('user')->where('workspace_id', $project->workspace_id)->get();
        return response()->json(['labels' => $labels, 'lists' => $lists, 'team_members' => $teamMembers]);
    }

    public function workspaceOtherData($workspace_id){
        $labels = Label::get();
        $teamMembers = TeamMember::with('user')->where('workspace_id', $workspace_id)->get();
        return response()->json(['labels' => $labels, 'team_members' => $teamMembers]);
    }

    private function clean($string) {
        $string = str_replace(' ', '-', $string);
        $string = filter_var($string, FILTER_SANITIZE_FULL_SPECIAL_CHARS);
        return preg_replace('/-+/', '-', $string);
    }

    public function destroy($id){
        $project = Project::where('id', $id)->first();
        $workspace_id = $project->workspace_id;
        if(!empty($project)){
            BoardList::where('project_id', $project->id)->delete();
            RecentProject::where('project_id', $project->id)->delete();
            StarredProject::where('project_id', $project->id)->delete();
            $tasks = Task::where('project_id', $project->id)->get();
            foreach ($tasks as $task){
                $attachments = Attachment::where('task_id', $task->id)->get();
                foreach ($attachments as $attachment){
                    if(!empty($attachment->path) && File::exists(public_path($attachment->path))){
                        File::delete(public_path($attachment->path));
                    }
                    $attachment->delete();
                }
                CheckList::where('task_id', $task->id)->delete();
                Timer::where('task_id', $task->id)->delete();
                Comment::where('task_id', $task->id)->delete();
                Assignee::where('task_id', $task->id)->delete();
                TaskLabel::where('task_id', $task->id)->delete();
                $task->delete();
            }
            $project->delete();
        }
        return Redirect::route('workspace.view', $workspace_id);
    }

    public function excelExport($project_id)
    {
        return (new TasksExport)->forProject($project_id)->download('tasks.xlsx', Excel::XLSX);
    }

    public function csvExport($project_id)
    {
        // New Version
        return (new TasksExport)->forProject($project_id)->download('tasks.csv', Excel::CSV);
        // New Version
//        $tasks = Task::where('project_id', $project_id)->limit(14)->orderBy('id', 'desc')->get();
//        $csvFileName = 'tasks.csv';
//
//        $headers = [
//            'Content-Type' => 'text/csv',
//            'Content-Disposition' => 'attachment; filename="' . $csvFileName . '"',
//        ];
//
//        $handle = fopen('php://output', 'w');
//        fputcsv($handle, ['ID', 'Title', 'Slug', 'Description', 'Board', 'Project Name', 'Due Date', 'Order', 'User ID','List ID', 'Created At']);
//
//        foreach ($tasks as $task) {
//            fputcsv($handle, [$task->id, $this->cleanCode($task->title), $this->cleanCode($task->slug),
//                $this->cleanCode(strip_tags($task->description)), $task->list ? $this->cleanCode($task->list->title) : null,
//                $task->project ? $this->cleanCode($task->project->title): null, $task->due_date,
//                $task->order, $task->user_id, $task->list_id, $task->created_at
//            ]);
//        }
//
//        fclose($handle);
//        return Response::make('', 200, $headers);
    }


}
